/*
 * Electric(tm) VLSI Design System
 *
 * File: dblibrary.c
 * Database library and lambda control module
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "global.h"
#include "database.h"

/* prototypes for local routines */
void db_scalefacet(NODEPROTO*, INTBIG, INTBIG);
void db_validatearcinst(ARCINST*);
void db_setarcinst(ARCINST*, INTBIG, INTBIG, INTBIG, INTBIG);
INTBIG db_muldivq(INTBIG, INTBIG, INTBIG);
INTBIG db_scaleunits(INTBIG *value, INTBIG num, INTBIG den);

/****************************** LIBRARIES ******************************/

/*
 * routine to allocate a library, places it in its own cluster, and return its
 * address.  The routine returns NOLIBRARY if allocation fails.
 */
LIBRARY *alloclibrary(void)
{
	REGISTER LIBRARY *lib;
	REGISTER TECHNOLOGY *tech;
	REGISTER CLUSTER *cluster;

	cluster = alloccluster("");
	if (cluster == NOCLUSTER) return((LIBRARY *)db_error(DBNOMEM|DBALLOCLIBRARY));
	lib = (LIBRARY *)emalloc((sizeof (LIBRARY)), cluster);
	if (lib == 0) return((LIBRARY *)db_error(DBNOMEM|DBALLOCLIBRARY));
	lib->cluster = cluster;

	/* allocate space for lambda array */
	lib->lambda = emalloc(((el_maxtech+1) * SIZEOFINTBIG), cluster);
	if (lib->lambda == 0) return((LIBRARY *)db_error(DBNOMEM|DBALLOCLIBRARY));
	lib->lambda[el_maxtech] = -1;

	lib->userbits = lib->temp1 = lib->temp2 = 0;
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
		lib->lambda[tech->techindex] = tech->deflambda;
	lib->numvar = 0;
	lib->libname = NOSTRING;
	lib->libfile = NOSTRING;
	lib->firstvar = NOVARIABLE;
	lib->firstnodeproto = NONODEPROTO;
	lib->firstcell = NOCELL;
	lib->curnodeproto = NONODEPROTO;
	lib->nextlibrary = NOLIBRARY;
	return(lib);
}

/*
 * routine to return library "lib" to the pool of free libraries
 */
void freelibrary(LIBRARY *lib)
{
	REGISTER CLUSTER *clus;

	if (lib == NOLIBRARY) return;
	if (lib->numvar != 0) db_freevars(&lib->firstvar, &lib->numvar);
	efree((char *)lib->lambda);
	clus = lib->cluster;
	efree((char *)lib);
	freecluster(clus);
}

/*
 * routine to create a new library and return its address.  The library name
 * is "name" and its disk file is "file".  If there is any error, NOLIBRARY
 * is returned.
 */
LIBRARY *newlibrary(char *name, char *file)
{
	REGISTER LIBRARY *lib;
	REGISTER char *ch, nc;
	REGISTER INTSML renamed;

	/* error checks */
	renamed = 0;
	for(ch = name; *ch != 0; ch++)
	{
		nc = *ch;
		if (nc == ' ') nc = '-';
		if (nc == '\t' || nc == ':') nc = '-';
		if (nc != *ch) renamed++;
		*ch = nc;
	}
	if (renamed != 0) ttyputerr("Warning: library renamed to '%s'", name);
	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
		if (namesame(name, lib->libname) == 0)
			return((LIBRARY *)db_error(DBBADLIB|DBNEWLIBRARY));

	/* create the library */
	lib = alloclibrary();
	if (lib == NOLIBRARY) return((LIBRARY *)db_error(DBNOMEM|DBNEWLIBRARY));
	(void)strcpy(lib->cluster->clustername, "lib:");
	(void)strncpy(&lib->cluster->clustername[4], name, 25);

	/* set library name and file */
	if (allocstring(&lib->libname, name, lib->cluster) != 0) return(NOLIBRARY);
	if (allocstring(&lib->libfile, file, lib->cluster) != 0) return(NOLIBRARY);

	/* set units */
	lib->userbits |= (((el_units & INTERNALUNITS) >> INTERNALUNITSSH) << LIBUNITSSH);

	/* link in the new library after the current library */
	if (el_curlib == NOLIBRARY)
	{
		/* this is the first library: make it the current one */
		lib->nextlibrary = NOLIBRARY;
		el_curlib = lib;
	} else
	{
		/* add this library to the list headed by "el_curlib" */
		lib->nextlibrary = el_curlib->nextlibrary;
		el_curlib->nextlibrary = lib;
	}

	/* tell constraint system about new library */
	(*el_curconstraint->newlib)(lib);

	/* report library address */
	return(lib);
}

/*
 * routine to set the current library (represented by the global "el_curlib")
 * to "lib"
 */
void selectlibrary(LIBRARY *lib)
{
	REGISTER LIBRARY *l, *lastlib;
	REGISTER TECHNOLOGY *tech;

	/* quit if already done */
	if (lib == NOLIBRARY) return;
	if (el_curlib == lib) return;

	/* unlink library from its current position */
	lastlib = NOLIBRARY;
	for(l = el_curlib; l != NOLIBRARY; l = l->nextlibrary)
	{
		if (l == lib) break;
		lastlib = l;
	}
	if (lastlib != NOLIBRARY) lastlib->nextlibrary = lib->nextlibrary;

	/* link in at the head of the list */
	lib->nextlibrary = el_curlib;
	el_curlib = lib;

	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
		db_scaletechnology(tech, el_curlib->lambda[tech->techindex], tech->deflambda);
}

void killlibrary(LIBRARY *lib)
{
	REGISTER LIBRARY *l, *lastlib;

	if (lib == el_curlib)
	{
		(void)db_error(DBBADLIB|DBKILLLIBRARY);
		return;
	}

	/* tell constraint system about killed library */
	(*el_curconstraint->killlib)(lib);

	/* unlink current library */
	lastlib = NOLIBRARY;
	for(l = el_curlib; l != NOLIBRARY; l = l->nextlibrary)
	{
		if (l == lib) break;
		lastlib = l;
	}
	if (lastlib != NOLIBRARY) lastlib->nextlibrary = lib->nextlibrary;

	/* kill the requested library */
	eraselibrary(lib);
	efree(lib->libfile);
	efree(lib->libname);
	freelibrary(lib);
}

/*
 * routine to erase the contents of a library of facets.
 * The index of the library is in "libindex".
 */
void eraselibrary(LIBRARY *lib)
{
	REGISTER NODEPROTO *np, *lnp;
	REGISTER PORTPROTO *pp, *lpt;
	REGISTER NODEINST *ni, *nni;
	REGISTER ARCINST *ai, *nai;
	REGISTER PORTARCINST *pi, *lpo;
	REGISTER PORTEXPINST *pe, *lpe;
	REGISTER NETWORK *net, *nnet;
	REGISTER CELL *c, *nextc;
	REGISTER INTSML i;

	/* see if this library exists */
	if (lib == NOLIBRARY) return;

	/* flush all batched changes */
	noundoallowed();

	for(i=0; i<el_maxaid; i++) 
		if (el_aids[i].eraselibrary != 0)
			(*el_aids[i].eraselibrary)(lib);

	/* erase the nodes, ports, arcs, and geometry modules in each nodeproto */
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		/* erase all arcs and nodes in this facet */
		for(ni = np->firstnodeinst; ni != NONODEINST; ni = nni)
		{
			nni = ni->nextnodeinst;

			/* remove nodeinst link from technology if it is primitive */
			if (ni->proto->primindex != 0)
			{
				if (ni->nextinst != NONODEINST) ni->nextinst->lastinst = ni->lastinst;
				if (ni->lastinst != NONODEINST) ni->lastinst->nextinst = ni->nextinst; else
					ni->proto->firstinst = ni->nextinst;
			}

			/* erase the portarcs on this nodeinst */
			for(pi = ni->firstportarcinst; pi != NOPORTARCINST; pi = lpo)
			{
				lpo = pi->nextportarcinst;
				freeportarcinst(pi);
			}

			/* erase the portexps on this nodeinst */
			for(pe = ni->firstportexpinst; pe != NOPORTEXPINST; pe = lpe)
			{
				lpe = pe->nextportexpinst;
				freeportexpinst(pe);
			}

			/* erase the nodeinst */
			freegeom(ni->geom);
			freenodeinst(ni);
		}
		for(ai = np->firstarcinst; ai != NOARCINST; ai = nai)
		{
			nai = ai->nextarcinst;
			freegeom(ai->geom);
			freearcinst(ai);
		}
		for(net = np->firstnetwork; net != NONETWORK; net = nnet)
		{
			void net_freenetwork(NETWORK*, NODEPROTO*);

			nnet = net->nextnetwork;
			net_freenetwork(net, np);
		}
		db_freertree(np->rtree);
	}

	/* now erase the portprotos and nodeprotos */
	for(np = lib->firstnodeproto; np != NONODEPROTO; np = lnp)
	{
		lnp = np->nextnodeproto;

		/* free the portproto entries */
		for(pp=np->firstportproto; pp != NOPORTPROTO; pp = lpt)
		{
			lpt = pp->nextportproto;
			efree(pp->protoname);
			freeportproto(pp);
		}

		/* free the nodeinst proto */
		freenodeproto(np);
	}
	lib->firstnodeproto = NONODEPROTO;
	lib->curnodeproto = NONODEPROTO;

	/* now erase the cells */
	for(c = lib->firstcell; c != NOCELL; c = nextc)
	{
		nextc = c->nextcell;
		efree(c->cellname);
		freecell(c);
	}
	lib->firstcell = NOCELL;

	/* now erase the library information */
	if (lib->numvar != 0) db_freevars(&lib->firstvar, &lib->numvar);
}

/****************************** LAMBDA ******************************/

/*
 * routine to change value of lambda from "oldlam" to "lam" and adjust the
 * world to agree with the new value.
 * if tech!=NOTECHNOLOGY just make a change to technology "tech" (scale the
 *                       technology and update the library/technology value).
 * if tech==NOTECHNOLOGY adjust all technologies to match the current library
 *                       and scale all libraries to match the new lambda.
 */
void changelambda(INTBIG oldlam, INTBIG lam, TECHNOLOGY *tech)
{
	REGISTER INTSML i;
	REGISTER NODEPROTO *np;
	REGISTER LIBRARY *lib;

	if (tech != NOTECHNOLOGY)
	{
		/* not scaling library: simply change this particular value */
		db_scaletechnology(tech, lam, tech->deflambda);
		el_curlib->lambda[tech->techindex] = lam;
		return;
	}

	/* scale the technology values in the current library */
	if (lam != oldlam)
		for(i=0; i < el_maxtech; i++)
			el_curlib->lambda[i] = muldiv(el_curlib->lambda[i], lam, oldlam);

	/* make all technologies agree with the current library */
	for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
		db_scaletechnology(tech, el_curlib->lambda[tech->techindex], tech->deflambda);

	/* no change to the library if lambda is the same */
	if (lam == oldlam) return;

	/* mark all facets in the library for scaling */
	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			np->temp1 = 0;

	/* scale all the facets */
	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
	{
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			db_scalefacet(np, lam, oldlam);
		lib->userbits |= LIBCHANGEDMAJOR;
	}
}

/*
 * routine to return the value of lambda to use for object "geom"
 */
INTBIG figurelambda(GEOM *geom)
{
	REGISTER ARCINST *ai;
	REGISTER NODEINST *ni;

	if (geom->entrytype == OBJARCINST)
	{
		ai = geom->entryaddr.ai;
		return(ai->proto->tech->deflambda);
	}
	ni = geom->entryaddr.ni;
	if (ni->proto->primindex != 0) return(ni->proto->tech->deflambda);
	return(el_curtech->deflambda);
}

/*
 * routine to recursively scale the contents of facet "np" by "lam/oldlam"
 */
void db_scalefacet(NODEPROTO *np, INTBIG lam, INTBIG oldlam)
{
	REGISTER NODEINST *ni;

	/* quit if the facet is already done */
	if (np->temp1 != 0) return;

	/* first look for sub-facets that are not yet scaled */
	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		if (ni->proto->primindex == 0) db_scalefacet(ni->proto, lam, oldlam);

	/* mark this facet "scaled" */
	np->temp1++;

	db_scaleonefacet(np, lam, oldlam);
}

/*
 * routine to scale facet "np" by the factor "num" / "denom"
 */
void db_scaleonefacet(NODEPROTO *np, INTBIG num, INTBIG denom)
{
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER INTBIG sizex, sizey, dx, dy, len;
	REGISTER INTSML i;
	REGISTER VARIABLE *var;

	for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
	{
		ni->lowx  = db_muldivq(ni->lowx,  num, denom);
		ni->highx = db_muldivq(ni->highx, num, denom);
		ni->lowy  = db_muldivq(ni->lowy,  num, denom);
		ni->highy = db_muldivq(ni->highy, num, denom);
		if (ni->proto->primindex == 0)
		{
			sizex = ni->proto->highx - ni->proto->lowx;
			sizey = ni->proto->highy - ni->proto->lowy;
			dx = ni->highx - ni->lowx - sizex;
			dy = ni->highy - ni->lowy - sizey;
			if (dx != 0 || dy != 0)
			{
				ttyputmsg("Facet %s, node %s size and position adjusted",
					describenodeproto(np), describenodeinst(ni));
				ni->lowx += dx/2;   ni->highx = ni->lowx + sizex;
				ni->lowy += dy/2;   ni->highy = ni->lowy + sizey;
			}
		} else
		{
			/* look for trace data on primitives */
			var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER|VISARRAY, el_trace);
			if (var != NOVARIABLE)
			{
				len = getlength(var);
				for(i=0; i<len; i++)
					((INTBIG *)var->addr)[i] = db_muldivq(((INTBIG *)var->addr)[i], num, denom);
			}
		}
		updategeom(ni->geom, np);
	}
	for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		ai->width = db_muldivq(ai->width, num, denom);
		ai->end[0].xpos = db_muldivq(ai->end[0].xpos, num, denom);
		ai->end[0].ypos = db_muldivq(ai->end[0].ypos, num, denom);
		ai->end[1].xpos = db_muldivq(ai->end[1].xpos, num, denom);
		ai->end[1].ypos = db_muldivq(ai->end[1].ypos, num, denom);
		ai->length = computedistance(ai->end[0].xpos, ai->end[0].ypos,
			ai->end[1].xpos, ai->end[1].ypos);
		db_validatearcinst(ai);
		updategeom(ai->geom, np);

		/* look for curvature data on primitives */
		var = getvalkey((INTBIG)ai, VARCINST, VINTEGER, el_arc_radius);
		if (var != NOVARIABLE) var->addr = db_muldivq(var->addr, num, denom);
	}
	db_boundfacet(np, &np->lowx, &np->highx, &np->lowy, &np->highy);
}

/*
 * routine to validate arcinst "ai" after being scaled.
 */
void db_validatearcinst(ARCINST *ai)
{
	REGISTER INTBIG wid;
	REGISTER INTSML inside0, inside1;
	INTBIG lx0,lx1,hx0,hx1, ly0,ly1,hy0,hy1, fx, fy, tx, ty;
	static POLYGON *poly0 = NOPOLYGON, *poly1 = NOPOLYGON;

	/* if nothing is outside port, quit */
	inside0 = db_stillinport(ai, 0, ai->end[0].xpos, ai->end[0].ypos);
	inside1 = db_stillinport(ai, 1, ai->end[1].xpos, ai->end[1].ypos);
	if (inside0 != 0 && inside1 != 0) return;

	/* make sure there is a polygon */
	if (poly0 == NOPOLYGON) poly0 = allocstaticpolygon(4, db_cluster);
	if (poly1 == NOPOLYGON) poly1 = allocstaticpolygon(4, db_cluster);

	/* get area of the ports */
	wid = ai->width - arcwidthoffset(ai);
	shapeportpoly(ai->end[0].nodeinst, ai->end[0].portarcinst->proto, poly0, 0);
	reduceportpoly(poly0, ai->end[0].nodeinst, ai->end[0].portarcinst->proto, wid);
	shapeportpoly(ai->end[1].nodeinst, ai->end[1].portarcinst->proto, poly1, 0);
	reduceportpoly(poly1, ai->end[1].nodeinst, ai->end[1].portarcinst->proto, wid);

	/* if arcinst is not fixed-angle, run it directly to the port centers */
	if ((ai->userbits&FIXANG) == 0)
	{
		getcenter(poly0, &fx, &fy);
		getcenter(poly1, &tx, &ty);
		db_setarcinst(ai, fx, fy, tx, ty);
		return;
	}

	/* get bounding boxes of polygons */
	getbbox(poly0, &lx0, &hx0, &ly0, &hy0);
	getbbox(poly1, &lx1, &hx1, &ly1, &hy1);

	/* if fixed-angle path runs between the ports, adjust the arcinst */
	if (arcconnects((INTSML)(((ai->userbits&AANGLE) >> AANGLESH) * 10), lx0,hx0, ly0,hy0, lx1,hx1,
		ly1,hy1, &fx,&fy, &tx,&ty) != 0)
	{
		closestpoint(poly0, &fx, &fy);
		closestpoint(poly1, &tx, &ty);
		db_setarcinst(ai, fx, fy, tx, ty);
		return;
	}

	/* give up and remove the constraint */
	getcenter(poly0, &fx, &fy);
	getcenter(poly1, &tx, &ty);
	ai->userbits &= ~FIXANG;
	db_setarcinst(ai, fx, fy, tx, ty);
	ttyputmsg("Facet %s, arc %s no longer fixed-angle",
		describenodeproto(ai->parent), describearcinst(ai));
}

void db_setarcinst(ARCINST *ai, INTBIG fx, INTBIG fy, INTBIG tx, INTBIG ty)
{
	/* check for null arcinst motion */
	if (fx == ai->end[0].xpos && fy == ai->end[0].ypos &&
		tx == ai->end[1].xpos && ty == ai->end[1].ypos) return;

	ai->end[0].xpos = fx;   ai->end[0].ypos = fy;
	ai->end[1].xpos = tx;   ai->end[1].ypos = ty;
	ai->length = computedistance(fx,fy, tx,ty);
	determineangle(ai);
	updategeom(ai->geom, ai->parent);
}

/* routine to compute (a*b/c) rounded to the nearest quarter "b" */
INTBIG db_muldivq(INTBIG a, INTBIG b, INTBIG c)
{
	REGISTER INTBIG j;

	j = muldiv(a, b, c) * 4;
	if (j < 0) j -= b/2; else j += b/2;
	return((j / b) * b / 4);
}

void db_scaletechnology(TECHNOLOGY *tech, INTBIG newlambda, INTBIG oldlambda)
{
	REGISTER NODEPROTO *np;
	REGISTER ARCPROTO *ap;
	REGISTER VARIABLE *var;

	/* change the default width of the primitive arc prototypes */
	for(ap = tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
	{
		ap->nominalwidth = muldiv(ap->nominalwidth, newlambda, oldlambda);
		var = getvalkey((INTBIG)ap, VARCPROTO, VINTEGER, el_arc_default_width);
		if (var != NOVARIABLE)
			var->addr = muldiv(var->addr, newlambda, oldlambda);
	}

	/* now change the default size of the primitive node prototypes */
	for(np = tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
	{
		np->lowx  = muldiv(np->lowx,  newlambda, oldlambda);
		np->highx = muldiv(np->highx, newlambda, oldlambda);
		np->lowy  = muldiv(np->lowy,  newlambda, oldlambda);
		np->highy = muldiv(np->highy, newlambda, oldlambda);
		var = getvalkey((INTBIG)np, VNODEPROTO, VINTEGER|VISARRAY, el_node_default_size);
		if (var != NOVARIABLE)
		{
			((INTBIG *)var->addr)[0] = muldiv(((INTBIG *)var->addr)[0], newlambda, oldlambda);
			((INTBIG *)var->addr)[1] = muldiv(((INTBIG *)var->addr)[1], newlambda, oldlambda);
		}
	}

	/* finally, reset the default lambda value */
	tech->deflambda = muldiv(tech->deflambda, newlambda, oldlambda);

	/* prevent zero lambda values */
	if (tech->deflambda <= 0) tech->deflambda = 1;
}

/*
 * Routine to change the internal units in library "whichlib" from "oldunits"
 * to "newunits".  If "whichlib" is NOLIBRARY, change the entire database (all
 * libraries and technologies).
 */
void changeinternalunits(LIBRARY *whichlib, INTSML oldunits, INTSML newunits)
{
	REGISTER INTBIG hardscaleerror, softscaleerror, len;
	INTBIG num, den;
	REGISTER INTSML i;
	REGISTER char *errortype, *errorlocation;
	REGISTER TECHNOLOGY *tech;
	REGISTER LIBRARY *lib;
	REGISTER NODEPROTO *np;
	REGISTER ARCPROTO *ap;
	REGISTER NODEINST *ni;
	REGISTER ARCINST *ai;
	REGISTER VARIABLE *var;
	REGISTER WINDOWPART *w;
	extern INTBIG us_alignment, us_edgealignment;

	hardscaleerror = softscaleerror = 0;
	db_getinternalunitscale(&num, &den, oldunits, newunits);
	if (whichlib == NOLIBRARY)
	{
		/* global change: set units */
		if ((oldunits&INTERNALUNITS) == newunits) return;
		el_units = (el_units & ~INTERNALUNITS) | newunits;

		/* scale all technologies */
		for(tech = el_technologies; tech != NOTECHNOLOGY; tech = tech->nexttechnology)
		{
			/* scale arc width */
			for(ap = tech->firstarcproto; ap != NOARCPROTO; ap = ap->nextarcproto)
			{
				hardscaleerror += db_scaleunits(&ap->nominalwidth, num, den);
				var = getvalkey((INTBIG)ap, VARCPROTO, VINTEGER, el_arc_default_width);
				if (var != NOVARIABLE)
					hardscaleerror += db_scaleunits(&var->addr, num, den);
			}

			/* scale node sizes */
			for(np = tech->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			{
				hardscaleerror += db_scaleunits(&np->lowx,  num, den);
				hardscaleerror += db_scaleunits(&np->highx, num, den);
				hardscaleerror += db_scaleunits(&np->lowy,  num, den);
				hardscaleerror += db_scaleunits(&np->highy, num, den);
				var = getvalkey((INTBIG)np, VNODEPROTO, VINTEGER|VISARRAY, el_node_default_size);
				if (var != NOVARIABLE)
				{
					hardscaleerror += db_scaleunits(&((INTBIG *)var->addr)[0], num, den);
					hardscaleerror += db_scaleunits(&((INTBIG *)var->addr)[1], num, den);
				}
			}

			/* finally, scale lambda */
			hardscaleerror += db_scaleunits(&tech->deflambda, num, den);
			if (tech->deflambda <= 0) tech->deflambda = 1;
		}

		/* scale lambda in libraries */
		for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
			for(i=0; i<el_maxtech; i++)
				hardscaleerror += db_scaleunits(&lib->lambda[i], num, den);

		/* scale all display windows */
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		{
			if (w->curnodeproto == NONODEPROTO) continue;
			softscaleerror += db_scaleunits(&w->screenlx, num, den);
			softscaleerror += db_scaleunits(&w->screenly, num, den);
			softscaleerror += db_scaleunits(&w->screenhx, num, den);
			softscaleerror += db_scaleunits(&w->screenhy, num, den);
			softscaleerror += db_scaleunits(&w->gridx, num, den);
			softscaleerror += db_scaleunits(&w->gridy, num, den);
		}

		/* scale miscellaneous factors */
		if (us_alignment != 0) softscaleerror += db_scaleunits(&us_alignment, num, den);
		if (us_edgealignment != 0) softscaleerror += db_scaleunits(&us_edgealignment, num, den);
	}

	/* scale all appropriate libraries */
	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
	{
		/* only scale requested library if request was made */
		if (whichlib != NOLIBRARY && whichlib != lib) continue;

		/* scale each facet */
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
		{
			/* scale nodes */
			for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
			{
				hardscaleerror += db_scaleunits(&ni->lowx,  num, den);
				hardscaleerror += db_scaleunits(&ni->highx, num, den);
				hardscaleerror += db_scaleunits(&ni->lowy,  num, den);
				hardscaleerror += db_scaleunits(&ni->highy, num, den);
				if (ni->proto->primindex != 0)
				{
					/* look for trace data on primitives */
					var = getvalkey((INTBIG)ni, VNODEINST, VINTEGER|VISARRAY, el_trace);
					if (var != NOVARIABLE)
					{
						len = getlength(var);
						for(i=0; i<len; i++)
							hardscaleerror += db_scaleunits(&((INTBIG *)var->addr)[i], num, den);
					}
				}
				updategeom(ni->geom, np);
			}

			/* scale arcs */
			for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
			{
				hardscaleerror += db_scaleunits(&ai->width, num, den);
				hardscaleerror += db_scaleunits(&ai->end[0].xpos, num, den);
				hardscaleerror += db_scaleunits(&ai->end[0].ypos, num, den);
				hardscaleerror += db_scaleunits(&ai->end[1].xpos, num, den);
				hardscaleerror += db_scaleunits(&ai->end[1].ypos, num, den);
				ai->length = computedistance(ai->end[0].xpos, ai->end[0].ypos,
					ai->end[1].xpos, ai->end[1].ypos);
				updategeom(ai->geom, np);

				/* look for curvature data */
				var = getvalkey((INTBIG)ai, VARCINST, VINTEGER, el_arc_radius);
				if (var != NOVARIABLE) hardscaleerror += db_scaleunits(&var->addr, num, den);
			}

			/* scale facet size */
			db_boundfacet(np, &np->lowx, &np->highx, &np->lowy, &np->highy);
		}
		lib->userbits = (lib->userbits & ~LIBUNITS) |
			(((newunits & INTERNALUNITS) >> INTERNALUNITSSH) << LIBUNITSSH);
		if (lib->firstnodeproto != NONODEPROTO) lib->userbits |= LIBCHANGEDMAJOR;
	}

	/* report any failures */
	if (hardscaleerror != 0 || softscaleerror != 0)
	{
		if (den > num) errortype = "roundoff"; else
			errortype = "overflow";
		if (hardscaleerror == 0) errorlocation = "display"; else
		{
			if (whichlib == NOLIBRARY) errorlocation = "database"; else
				errorlocation = "library";
		}
		ttyputerr("Change caused %s errors in %s", errortype, errorlocation);
		if (hardscaleerror != 0)
		{
			if (whichlib == NOLIBRARY)
			{
				ttyputmsg("Recommend check of database for validity and keep old library files");
				for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
					lib->userbits &= ~READFROMDISK;
			} else
			{
				ttyputmsg("Recommend check of database for validity and keep old library file");
				whichlib->userbits &= ~READFROMDISK;
			}
		} else
		{
			ttyputmsg("Recommend redisplay");
		}
	}
}

/*
 * Routine to scale "value" by "num"/"den".  Returns zero if scale worked,
 * nonzero if overflow/underflow occurred.
 */
INTBIG db_scaleunits(INTBIG *value, INTBIG num, INTBIG den)
{
	REGISTER INTBIG orig, scaled, reconstructed;

	orig = *value;
	scaled = muldiv(orig, num, den);
	*value = scaled;
	reconstructed = muldiv(scaled, den, num);
	if (orig == reconstructed) return(0);
	return(1);
}
